<?php
/**
 * Plugin Name: MU Code Editor
 * Description: Manage and edit Must-Use plugins with a simple grid + editor. Toggle enable/disable by moving files between mu-plugins and a sandbox folder.
 * Version: 1.0.4
 * Author: The Code Rebel
 */

if (!defined('ABSPATH')) { exit; }

/* ---------- PHP 7 polyfills ---------- */
if (!function_exists('str_starts_with')) {
    function str_starts_with($haystack, $needle) {
        return $needle === '' || strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}
if (!function_exists('str_ends_with')) {
    function str_ends_with($haystack, $needle) {
        if ($needle === '') return true;
        $n = strlen($needle);
        return substr($haystack, -$n) === $needle;
    }
}

/* ---------- Plugin ---------- */
class TCR_MU_Editor {
    const SLUG = 'tcr-mu-editor';
    const CAP  = 'manage_options';

    private $live_dir;
    private $sandbox_dir;

    public function __construct() {
        $this->live_dir    = WP_CONTENT_DIR . '/mu-plugins';
        $this->sandbox_dir = WP_CONTENT_DIR . '/mu-plugins-sandbox';

        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'maybe_boot_dirs']);
        add_action('admin_post_tcr_mu_action', [$this, 'handle_post']);
        add_action('admin_enqueue_scripts', [$this, 'assets']);
    }

    public function menu() {
        add_submenu_page(
            'plugins.php',
            __('MU Code Editor', 'tcr'),
            __('MU Code Editor', 'tcr'),
            self::CAP,
            self::SLUG,
            [$this, 'render_page']
        );
    }

    public function assets($hook) {
        if ($hook !== 'plugins_page_' . self::SLUG && $hook !== 'toplevel_page_' . self::SLUG) return;
        $css = '
        .tcr-wrap{max-width:1200px}
        .tcr-card{background:#fff;border:1px solid #dcdcde;border-radius:10px;padding:16px;margin:0 0 16px}
        .tcr-grid table{width:100%;border-collapse:collapse}
        .tcr-grid th,.tcr-grid td{padding:10px;border-bottom:1px solid #eee;vertical-align:top}
        .tcr-status-badge{display:inline-block;padding:2px 8px;border-radius:999px;font-size:11px}
        .tcr-live{background:#e6ffed;border:1px solid #b4f5c2}
        .tcr-sandbox{background:#fff4e5;border:1px solid #ffd9a8}
        .tcr-actions form{display:inline}
        .tcr-mono{font-family:Menlo,Consolas,monospace}
        .tcr-editor textarea{width:100%;min-height:560px;font-family:Menlo,Consolas,monospace;font-size:13px}
        .tcr-row-meta{color:#666;font-size:12px}
        .tcr-btn{display:inline-block;margin-right:6px}
        .tcr-input-sm{width:320px}
        ';
        wp_add_inline_style('wp-components', $css);
    }

    public function maybe_boot_dirs() {
        if (!is_dir($this->live_dir))    wp_mkdir_p($this->live_dir);
        if (!is_dir($this->sandbox_dir)) wp_mkdir_p($this->sandbox_dir);
        foreach ([$this->live_dir, $this->sandbox_dir] as $dir) {
            $idx = trailingslashit($dir) . 'index.html';
            if (!file_exists($idx)) @file_put_contents($idx, '');
        }
    }

    private function check_caps() {
        if (!current_user_can(self::CAP)) {
            wp_die(__('Sorry, you are not allowed to access this page.', 'tcr'), 403);
        }
    }

    private function nonce($action) { return wp_nonce_field($action, '_tcr_nonce', true, false); }

    private function verify_nonce($action) {
        if (empty($_POST['_tcr_nonce']) || !wp_verify_nonce($_POST['_tcr_nonce'], $action)) {
            wp_die(__('Security check failed.', 'tcr'), 403);
        }
    }

    private function sanitize_filename($name) {
        $name = wp_unslash($name);
        $name = strtolower(trim($name));
        $name = preg_replace('/[^a-z0-9_\-\.]/', '-', $name);
        $name = basename($name);
        if (!str_ends_with($name, '.php')) $name .= '.php';
        return $name;
    }

    private function is_in_dir($path, $dir) {
        $real = realpath($path);
        $rdir = realpath($dir);
        return $real && $rdir && str_starts_with($real, $rdir);
    }

    private function mu_files() {
        $rows = [];
        foreach ([['dir'=>$this->live_dir,'status'=>'live'], ['dir'=>$this->sandbox_dir,'status'=>'sandbox']] as $set) {
            if (!is_dir($set['dir'])) continue;
            $files = glob(trailingslashit($set['dir']) . '*.php') ?: [];
            foreach ($files as $file) $rows[] = $this->file_row($file, $set['status']);
        }
        usort($rows, function($a,$b){
            if ($a['status'] !== $b['status']) return $a['status'] === 'live' ? -1 : 1;
            return strcmp($a['name'], $b['name']);
        });
        return $rows;
    }

    private function file_row($path, $status) {
        $data = $this->read_plugin_header($path);
        return [
            'name'        => basename($path),
            'path'        => $path,
            'status'      => $status,
            'description' => $data['Description'] ?? '',
            'plugin_name' => $data['Plugin Name'] ?? '',
            'version'     => $data['Version'] ?? '',
            'modified'    => file_exists($path) ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), filemtime($path)) : '',
            'size'        => file_exists($path) ? size_format(filesize($path)) : '',
        ];
    }

    private function read_plugin_header($file) {
        $headers = ['Plugin Name'=>'Plugin Name','Description'=>'Description','Version'=>'Version','Author'=>'Author'];
        $contents = file_exists($file) ? file_get_contents($file, false, null, 0, 8192) : '';
        $data = [];
        foreach ($headers as $field=>$label) {
            if (preg_match('/^' . preg_quote($label, '/') . ':\s*(.+)$/mi', $contents, $m)) $data[$field] = trim($m[1]);
        }
        return $data;
    }

    public function handle_post() {
        $this->check_caps();
        $action = isset($_POST['tcr_action']) ? sanitize_key($_POST['tcr_action']) : '';
        $redirect = admin_url('admin.php?page=' . self::SLUG);

        switch ($action) {
            case 'create':
                $this->verify_nonce('tcr_create');
                $filename = $this->sanitize_filename($_POST['filename'] ?? '');
                $target = trailingslashit($this->sandbox_dir) . $filename;
                if (file_exists($target)) { $redirect = add_query_arg('tcr_msg', rawurlencode('File already exists.'), $redirect); break; }
                $code = wp_unslash($_POST['code'] ?? '');
                if (trim($code) === '') {
                    $code = "/**\n * Plugin Name: {$filename}\n * Description: MU sandbox plugin created by MU Code Editor.\n */\n";
                }
                if (!str_starts_with(ltrim($code), '<?php')) $code = "<?php\n" . $code;
                wp_mkdir_p($this->sandbox_dir);
                file_put_contents($target, $code);
                $redirect = add_query_arg('tcr_msg', rawurlencode('Created in sandbox.'), $redirect) . '#tcr-editor';
                break;

            case 'save':
                $this->verify_nonce('tcr_save');
                $where = ($_POST['where'] ?? '') === 'live' ? $this->live_dir : $this->sandbox_dir;
                $filename = $this->sanitize_filename($_POST['filename'] ?? '');
                $path = trailingslashit($where) . $filename;
                if (!file_exists($path) || !$this->is_in_dir($path, $where)) { wp_die(__('Invalid file.', 'tcr'), 400); }
                $code = wp_unslash($_POST['code'] ?? '');
                if (!str_starts_with(ltrim($code), '<?php')) $code = "<?php\n" . $code;
                file_put_contents($path, $code);
                $redirect = add_query_arg(['tcr_msg'=>rawurlencode('Saved.'),'edit'=>$filename,'where'=>($_POST['where'] ?? 'sandbox')], $redirect) . '#tcr-editor';
                break;

            case 'enable':
            case 'disable':
                $this->verify_nonce($action === 'enable' ? 'tcr_enable' : 'tcr_disable');
                $filename = $this->sanitize_filename($_POST['filename'] ?? '');
                if ($action === 'enable') { $src = trailingslashit($this->sandbox_dir) . $filename; $dst = trailingslashit($this->live_dir) . $filename; }
                else { $src = trailingslashit($this->live_dir) . $filename; $dst = trailingslashit($this->sandbox_dir) . $filename; }
                if (!file_exists($src) || !$this->is_in_dir($src, dirname($src))) { wp_die(__('Invalid file.', 'tcr'), 400); }
                @rename($src, $dst);
                $redirect = add_query_arg('tcr_msg', rawurlencode($action === 'enable' ? 'Enabled (moved to mu-plugins).' : 'Disabled (moved to sandbox).'), $redirect) . '#tcr-editor';
                break;

            case 'delete':
                $this->verify_nonce('tcr_delete');
                $where = ($_POST['where'] ?? '') === 'live' ? $this->live_dir : $this->sandbox_dir;
                $filename = $this->sanitize_filename($_POST['filename'] ?? '');
                $path = trailingslashit($where) . $filename;
                if (!file_exists($path) || !$this->is_in_dir($path, $where)) { wp_die(__('Invalid file.', 'tcr'), 400); }
                @unlink($path);
                $redirect = add_query_arg('tcr_msg', rawurlencode('Deleted.'), $redirect) . '#tcr-editor';
                break;

            case 'download':
                $this->verify_nonce('tcr_download');
                $where = ($_POST['where'] ?? '') === 'live' ? $this->live_dir : $this->sandbox_dir;
                $filename = $this->sanitize_filename($_POST['filename'] ?? '');
                $path = trailingslashit($where) . $filename;
                if (!file_exists($path) || !$this->is_in_dir($path, $where)) { wp_die(__('Invalid file.', 'tcr'), 400); }
                nocache_headers();
                header('Content-Type: application/x-php');
                header('Content-Disposition: attachment; filename="'. $filename .'"');
                header('Content-Length: ' . filesize($path));
                readfile($path);
                exit;
        }
        wp_safe_redirect($redirect);
        exit;
    }

    public function render_page() {
        $this->check_caps();

        $rows  = $this->mu_files();
        $msg   = isset($_GET['tcr_msg']) ? sanitize_text_field(wp_unslash($_GET['tcr_msg'])) : '';
        $edit  = isset($_GET['edit']) ? $this->sanitize_filename($_GET['edit']) : '';
        $where = isset($_GET['where']) && $_GET['where'] === 'live' ? 'live' : 'sandbox';

        $editing_path = '';
        if ($edit) { $editing_path = trailingslashit($where === 'live' ? $this->live_dir : $this->sandbox_dir) . $edit; }

        echo '<div class="wrap tcr-wrap">';
        echo '<h1 class="wp-heading-inline">'.esc_html__('MU Code Editor', 'tcr').'</h1>';
        if ($msg) { echo '<div class="notice notice-success is-dismissible"><p>'.esc_html($msg).'</p></div>'; }

        // Unified Editor (Create or Edit)
        echo '<a id="tcr-editor"></a>';
        echo '<div class="tcr-card tcr-editor">';
        if ($edit && file_exists($editing_path)) {
            $code = file_get_contents($editing_path);
            echo '<h2>'.esc_html(sprintf(__('Editing: %s', 'tcr'), $edit)).'</h2>';
            echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
            echo '<input type="hidden" name="action" value="tcr_mu_action" />';
            echo '<input type="hidden" name="tcr_action" value="save" />';
            echo '<input type="hidden" name="filename" value="'.esc_attr($edit).'" />';
            echo '<input type="hidden" name="where" value="'.esc_attr($where).'" />';
            echo $this->nonce('tcr_save');
            echo '<p><strong>'.esc_html__('Filename:', 'tcr').'</strong> <span class="tcr-mono">'.esc_html($edit).'</span> ';
            echo '<span class="tcr-row-meta">('.esc_html($where === 'live' ? __('Enabled (mu-plugins)', 'tcr') : __('Disabled (sandbox)', 'tcr')).')</span></p>';
            echo '<p><textarea name="code" class="large-text code tcr-mono" rows="22">'.esc_textarea($code).'</textarea></p>';
            submit_button(__('Save Changes', 'tcr'));
            $clear_url = remove_query_arg(['edit','where','tcr_msg'], admin_url('admin.php?page='.self::SLUG.'#tcr-editor'));
            echo '<a class="button button-link-delete" href="'.esc_url($clear_url).'">'.esc_html__('Cancel Editing', 'tcr').'</a>';
            echo '</form>';
        } else {
            echo '<h2>'.esc_html__('Create New (goes to Sandbox)', 'tcr').'</h2>';
            echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
            echo '<input type="hidden" name="action" value="tcr_mu_action" />';
            echo '<input type="hidden" name="tcr_action" value="create" />';
            echo $this->nonce('tcr_create');
            echo '<p><label>'.esc_html__('Filename (.php will be appended if omitted):','tcr').' ';
            echo '<input class="regular-text tcr-input-sm" type="text" name="filename" placeholder="my-mu-helper.php" required></label></p>';
            echo '<p><label>'.esc_html__('Initial Code (optional):','tcr').'</label></p>';
            echo '<p><textarea name="code" rows="16" class="large-text code tcr-mono" placeholder="/** Plugin Name: My MU Helper */"></textarea></p>';
            submit_button(__('Create in Sandbox', 'tcr'));
            echo '</form>';
        }
        echo '</div>';

        // MU Files grid
        echo '<div class="tcr-card">';
        echo '<h2>'.esc_html__('MU Files', 'tcr').'</h2>';
        echo '<div class="tcr-grid">';
        echo '<table class="widefat striped">';
        echo '<thead><tr>';
        echo '<th>'.esc_html__('File', 'tcr').'</th>';
        echo '<th>'.esc_html__('Plugin Name / Description', 'tcr').'</th>';
        echo '<th>'.esc_html__('Status', 'tcr').'</th>';
        echo '<th>'.esc_html__('Size', 'tcr').'</th>';
        echo '<th>'.esc_html__('Modified', 'tcr').'</th>';
        echo '<th>'.esc_html__('Actions', 'tcr').'</th>';
        echo '</tr></thead><tbody>';

        $rows = $this->mu_files();
        if (empty($rows)) {
            echo '<tr><td colspan="6">'.esc_html__('No MU files found.', 'tcr').'</td></tr>';
        } else {
            foreach ($rows as $r) {
                $badge_class = $r['status']==='live' ? 'tcr-status-badge tcr-live' : 'tcr-status-badge tcr-sandbox';
                $status_label = $r['status']==='live' ? __('Enabled', 'tcr') : __('Disabled (Sandbox)', 'tcr');
                echo '<tr>';
                echo '<td><strong class="tcr-mono">'.esc_html($r['name']).'</strong><div class="tcr-row-meta">'.esc_html(dirname($r['path'])).'</div></td>';
                echo '<td><strong>'.esc_html($r['plugin_name'] ?: '(no header)').'</strong><br/><span>'.esc_html($r['description']).'</span></td>';
                echo '<td><span class="'.$badge_class.'">'.esc_html($status_label).'</span></td>';
                echo '<td>'.esc_html($r['size']).'</td>';
                echo '<td>'.esc_html($r['modified']).'</td>';
                echo '<td class="tcr-actions">';
                $edit_url = add_query_arg(['page'=>self::SLUG,'edit'=>$r['name'],'where'=>$r['status']], admin_url('admin.php')) . '#tcr-editor';
                echo '<a class="button button-small tcr-btn" href="'.esc_url($edit_url).'">'.esc_html__('Edit', 'tcr').'</a>';
                if ($r['status'] === 'sandbox') {
                    echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
                    echo '<input type="hidden" name="action" value="tcr_mu_action"/>';
                    echo '<input type="hidden" name="tcr_action" value="enable"/>';
                    echo '<input type="hidden" name="filename" value="'.esc_attr($r['name']).'"/>';
                    echo $this->nonce('tcr_enable');
                    echo '<button class="button button-small tcr-btn" type="submit">'.esc_html__('Enable', 'tcr').'</button>';
                    echo '</form>';
                } else {
                    echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
                    echo '<input type="hidden" name="action" value="tcr_mu_action"/>';
                    echo '<input type="hidden" name="tcr_action" value="disable"/>';
                    echo '<input type="hidden" name="filename" value="'.esc_attr($r['name']).'"/>';
                    echo $this->nonce('tcr_disable');
                    echo '<button class="button button-small tcr-btn" type="submit">'.esc_html__('Disable', 'tcr').'</button>';
                    echo '</form>';
                }
                echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'">';
                echo '<input type="hidden" name="action" value="tcr_mu_action"/>';
                echo '<input type="hidden" name="tcr_action" value="download"/>';
                echo '<input type="hidden" name="where" value="'.esc_attr($r['status']).'"/>';
                echo '<input type="hidden" name="filename" value="'.esc_attr($r['name']).'"/>';
                echo $this->nonce('tcr_download');
                echo '<button class="button button-small tcr-btn" type="submit">'.esc_html__('Download', 'tcr').'</button>';
                echo '</form>';
                echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'" onsubmit="return confirm(\'Delete this file?\')">';
                echo '<input type="hidden" name="action" value="tcr_mu_action"/>';
                echo '<input type="hidden" name="tcr_action" value="delete"/>';
                echo '<input type="hidden" name="where" value="'.esc_attr($r['status']).'"/>';
                echo '<input type="hidden" name="filename" value="'.esc_attr($r['name']).'"/>';
                echo $this->nonce('tcr_delete');
                echo '<button class="button button-small tcr-btn" type="submit">'.esc_html__('Delete', 'tcr').'</button>';
                echo '</form>';
                echo '</td>';
                echo '</tr>';
            }
        }
        echo '</tbody></table>';
        echo '</div></div>'; // grid + card
        echo '<p class="description">'.esc_html__('Tip: MU plugins load on every request. Keep the sandbox disabled state for experiments. Enabling moves the file into wp-content/mu-plugins/.', 'tcr').'</p>';
        echo '</div>'; // wrap
    }
}
new TCR_MU_Editor();
