#include <FS.h>
#include <LittleFS.h>
#include <TFT_eSPI.h>
#include <XPT2046_Touchscreen.h>
#include <SPI.h>

static const uint32_t BAUD = 115200;
static const bool SHOW_LABELS = true;

// Touch (VSPI)
#define T_CS    33
#define T_CLK   25
#define T_MOSI  32
#define T_MISO  39

TFT_eSPI tft;
SPIClass touchSPI(VSPI);
XPT2046_Touchscreen ts(T_CS);

// Layout
#define SCREEN_W 320
#define SCREEN_H 240
#define BTN_ROWS 2
#define BTN_COLS 3
#define BTN_GAP  12
#define BTN_RADIUS 10

struct Button { int x, y, size; uint16_t color; const char* label; };
Button buttons[6];
const char* LABELS[6] = {"Dr","ea","m","S","ta","ck"};
int BTN_SIZE = 90;

// Touch mapping
constexpr int X_MIN=200, X_MAX=3700, Y_MIN=240, Y_MAX=3800;
int mapX(int raw){ return constrain(map(raw, X_MIN, X_MAX, 0, SCREEN_W), 0, SCREEN_W-1); }
int mapY(int raw){ return constrain(map(raw, Y_MIN, Y_MAX, 0, SCREEN_H), 0, SCREEN_H-1); }

// Upload state
enum RxState { IDLE, RX_IMG };
RxState  rxState = IDLE;
volatile bool UPLOAD_BUSY = false;
uint8_t  rxIdx = 0;
uint16_t rxW = 0, rxH = 0;
char     rxKind = 'N';
uint32_t rxTotal = 0, rxBytesLeft = 0, rxLastByteMs = 0;
fs::File rxFile; String rxTmpPath, rxFinalPath;

// Utils
static inline uint16_t dim565(uint16_t c) {
  uint8_t r = ((c >> 11) & 0x1F) << 3;
  uint8_t g = ((c >> 5)  & 0x3F) << 2;
  uint8_t b = ( c        & 0x1F) << 3;
  r >>= 1; g >>= 1; b >>= 1;
  return ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
}
void drawLabelCentered(int cx, int cy, const char* txt, uint16_t fg, uint16_t bg) {
  if (!SHOW_LABELS || !txt) return;
  tft.setTextDatum(MC_DATUM);
  tft.setTextColor(fg, bg);
  tft.drawString(txt, cx, cy, 4);
}

String imgPath(uint8_t idx, char kind){ char b[20]; snprintf(b,sizeof(b),"/btn%u_%c.raw",idx,kind); return String(b); }
uint32_t expectedSize(){ return (uint32_t)BTN_SIZE * BTN_SIZE * 2; }
bool validIcon(const String& p){
  if (!LittleFS.exists(p)) return false;
  fs::File f = LittleFS.open(p,"r"); if (!f) return false;
  uint32_t sz = f.size(); f.close();
  if (sz != expectedSize()) { LittleFS.remove(p); return false; }
  return true;
}
bool hasImage(uint8_t idx, char k){ return validIcon(imgPath(idx,k)); }

bool drawRawImage(const String& path, int16_t x, int16_t y, int w, int h){
  fs::File f = LittleFS.open(path,"r"); if(!f) return false;
  const size_t lineBytes = w*2;
  static uint8_t* line=nullptr; static size_t cap=0;
  if (cap < lineBytes){ if(line) free(line); line=(uint8_t*)malloc(lineBytes); cap=lineBytes; if(!line){ f.close(); return false; } }
  tft.setSwapBytes(true);
  for (int r=0;r<h;r++){
    if (f.read(line,lineBytes)!=(int)lineBytes){ f.close(); return false; }
    tft.pushImage(x, y+r, w, 1, (uint16_t*)line);
  }
  f.close(); return true;
}

bool generatePressedFromNormal(uint8_t idx){
  String pn=imgPath(idx,'N'), pp=imgPath(idx,'P');
  if (!LittleFS.exists(pn) || LittleFS.exists(pp)) return false;
  fs::File in=LittleFS.open(pn,"r"), out=LittleFS.open(pp,"w");
  if(!in||!out){ if(in) in.close(); if(out) out.close(); return false; }
  const size_t lineBytes=BTN_SIZE*2; uint8_t* line=(uint8_t*)malloc(lineBytes); if(!line){ in.close(); out.close(); return false; }
  for (int r=0;r<BTN_SIZE;r++){
    if (in.read(line,lineBytes)!=(int)lineBytes){ free(line); in.close(); out.close(); return false; }
    for (int i=0;i<(int)lineBytes;i+=2){
      uint16_t px = line[i] | (line[i+1]<<8);
      uint16_t R=(px>>11)&0x1F, G=(px>>5)&0x3F, B=px&0x1F;
      R>>=1; G>>=1; B>>=1; uint16_t q=(R<<11)|(G<<5)|B;
      line[i]=(uint8_t)(q&0xFF); line[i+1]=(uint8_t)(q>>8);
    }
    out.write(line,lineBytes);
  }
  free(line); in.close(); out.close(); return true;
}

void drawButtonBase(const Button& b, bool pressed, uint8_t idx1){
  if (pressed && !hasImage(idx1,'P') && hasImage(idx1,'N')) generatePressedFromNormal(idx1);
  char want = pressed ? 'P':'N';
  if (hasImage(idx1,want)){ drawRawImage(imgPath(idx1,want), b.x,b.y,b.size,b.size); return; }
  if (!pressed && hasImage(idx1,'N')){ drawRawImage(imgPath(idx1,'N'), b.x,b.y,b.size,b.size); return; }
  uint16_t fill = pressed ? dim565(b.color) : b.color;
  tft.fillRoundRect(b.x,b.y,b.size,b.size,BTN_RADIUS,fill);
  if (pressed) tft.drawRoundRect(b.x,b.y,b.size,b.size,BTN_RADIUS,TFT_WHITE);
  int cx=b.x+b.size/2, cy=b.y+b.size/2; uint16_t fg=TFT_WHITE;
  if (((b.color>>11)&0x1F)>20 && ((b.color>>5)&0x3F)>40) fg=TFT_BLACK;
  drawLabelCentered(cx,cy,LABELS[idx1-1],fg,fill);
}
void drawButtonNormal (const Button& b,uint8_t i){ drawButtonBase(b,false,i); }
void drawButtonPressed(const Button& b,uint8_t i){ drawButtonBase(b,true ,i); }

int hitTest(int tx,int ty){
  for(int i=0;i<6;i++){ const Button &b=buttons[i]; if(tx>=b.x&&tx<=b.x+b.size&&ty>=b.y&&ty<=b.y+b.size) return i; }
  return -1;
}

// Protocol
void startImg(uint8_t idx,uint16_t w,uint16_t h,char kind){
  if (idx<1||idx>6||w!=BTN_SIZE||h!=BTN_SIZE||(kind!='N'&&kind!='P')){
    Serial.printf("ERR PARAM idx=%u w=%u h=%u kind=%c (BTN_SIZE=%d)\n",idx,w,h,kind,BTN_SIZE); return;
  }
  rxIdx=idx; rxW=w; rxH=h; rxKind=kind; rxTotal=rxBytesLeft=(uint32_t)w*h*2;
  rxFinalPath = imgPath(idx, kind);
  rxTmpPath   = rxFinalPath + ".tmp";
  if (LittleFS.exists(rxTmpPath)) LittleFS.remove(rxTmpPath);
  rxFile = LittleFS.open(rxTmpPath, "w");
  if(!rxFile){ Serial.println("ERR OPEN"); return; }

  rxState=RX_IMG; UPLOAD_BUSY=true; rxLastByteMs=millis();
  //Serial.printf("OK IMG BTN%u %ux%u %c need=%lu\n",idx,w,h,kind,(unsigned long)rxTotal);
  Serial.println("OK IMG");

}
void feedImg() {
  int avail = Serial.available();
  if (avail <= 0) {
    if (rxBytesLeft > 0 && (millis() - rxLastByteMs) > 15000) { // 15s timeout
      rxFile.close();
      if (LittleFS.exists(rxTmpPath)) LittleFS.remove(rxTmpPath);
      rxState = IDLE;
      UPLOAD_BUSY = false;
      Serial.println("ERR TIMEOUT");
    }
    return;
  }

  static uint8_t buf[2048];  // bigger buffer than before
  int want = (rxBytesLeft < (int)sizeof(buf)) ? rxBytesLeft : (int)sizeof(buf);
  if (avail < want) want = avail;

  int got = Serial.readBytes((char*)buf, want);
  if (got > 0) {
    size_t w = rxFile.write(buf, got);
    if (w != (size_t)got) {
      rxFile.close();
      if (LittleFS.exists(rxTmpPath)) LittleFS.remove(rxTmpPath);
      rxState = IDLE;
      UPLOAD_BUSY = false;
      Serial.println("ERR FS");
      return;
    }

    rxBytesLeft -= got;
    rxLastByteMs = millis();

    // Debug progress (comment out when stable)
    Serial.printf("# Received %lu / %lu\n",
                  (unsigned long)(rxTotal - rxBytesLeft),
                  (unsigned long)rxTotal);

    if (rxBytesLeft == 0) {
      rxFile.close();
      if (LittleFS.exists(rxFinalPath)) LittleFS.remove(rxFinalPath);
      LittleFS.rename(rxTmpPath, rxFinalPath);
      rxState = IDLE;
      UPLOAD_BUSY = false;
      Serial.println("OK DONE");
      drawButtonNormal(buttons[rxIdx - 1], rxIdx);
    }
  }
}
void handleLine(const String& line){
  if (line=="HELLO"){ Serial.printf("HELLO BTN=6 SIZE=%d\n", BTN_SIZE); return; }
  if (line=="STATUS"){ Serial.println(UPLOAD_BUSY?"BUSY":"IDLE"); return; }
  if (line=="ABORT"){
    if (rxState==RX_IMG){ rxFile.close(); if(LittleFS.exists(rxTmpPath)) LittleFS.remove(rxTmpPath); rxState=IDLE; UPLOAD_BUSY=false; }
    Serial.println("OK ABORT"); return;
  }
  if (line=="CLEAR"){
    for(int i=1;i<=6;i++){ String n=imgPath(i,'N'), p=imgPath(i,'P'); if(LittleFS.exists(n)) LittleFS.remove(n); if(LittleFS.exists(p)) LittleFS.remove(p); }
    Serial.println("OK CLEAR");
    for(int i=0;i<6;i++) drawButtonNormal(buttons[i], i+1);
    return;
  }
  if (line.startsWith("IMG ")){
    uint8_t idx=0; uint16_t w=0,h=0; char kind='N';
    int m=sscanf(line.c_str(),"IMG %hhu %hu %hu %c",&idx,&w,&h,&kind);
    if (m<3){ Serial.println("ERR SYN"); return; }
    if (m<4) kind='N';
    startImg(idx,w,h,kind); return;
  }
  Serial.println("ERR CMD");
}

void setup(){
  Serial.begin(BAUD);
  LittleFS.begin();

  tft.init(); tft.setRotation(1); tft.fillScreen(TFT_BLACK);

  pinMode(T_CS,OUTPUT); digitalWrite(T_CS,HIGH);
  touchSPI.begin(T_CLK, T_MISO, T_MOSI, T_CS);
  ts.begin(touchSPI); ts.setRotation(1);

  int s = (SCREEN_W - (BTN_COLS+1)*BTN_GAP)/BTN_COLS;
  int maxH = (SCREEN_H - (BTN_ROWS+1)*BTN_GAP)/BTN_ROWS;
  if (s>maxH) s=maxH; BTN_SIZE=s;

  for(int i=1;i<=6;i++){ validIcon(imgPath(i,'N')); validIcon(imgPath(i,'P')); }

  uint16_t colours[6]={TFT_RED,TFT_GREEN,TFT_BLUE,TFT_ORANGE,TFT_PURPLE,TFT_DARKGREY};
  int totalH = BTN_ROWS*s + (BTN_ROWS+1)*BTN_GAP;
  int yOff = (SCREEN_H-totalH)/2;

  for(int r=0,idx=0;r<BTN_ROWS;r++){
    for(int c=0;c<BTN_COLS;c++,idx++){
      Button &b=buttons[idx];
      b.x=BTN_GAP + c*(s+BTN_GAP);
      b.y=yOff + BTN_GAP + r*(s+BTN_GAP);
      b.size=s; b.color=colours[idx]; b.label=LABELS[idx];
      drawButtonNormal(b, idx+1);
    }
  }
  Serial.println("READY");
}

void loop(){
  if (rxState==RX_IMG) feedImg();
  else if (Serial.available()){
    while (Serial.available()){
      String line = Serial.readStringUntil('\n'); line.trim();
      if (line.length()) handleLine(line);
    }
  }

  if (UPLOAD_BUSY){ delay(5); return; }
  if (!ts.touched()){ delay(5); return; }
  TS_Point p=ts.getPoint(); if (p.z<10){ delay(5); return; }
  int tx=mapX(p.x), ty=mapY(p.y); int idx=hitTest(tx,ty);
  if (idx>=0){
    drawButtonPressed(buttons[idx], idx+1);
    Serial.printf("P,%d\nBTN:%d\n", idx+1, idx+1);
    while(ts.touched()){
      TS_Point q=ts.getPoint(); int qx=mapX(q.x), qy=mapY(q.y);
      if (qx<buttons[idx].x || qx>buttons[idx].x+buttons[idx].size || qy<buttons[idx].y || qy>buttons[idx].y+buttons[idx].size) break;
      delay(10);
    }
    drawButtonNormal(buttons[idx], idx+1);
    Serial.printf("R,%d\n", idx+1);
    while(ts.touched()) delay(10);
  }
}
