<?php
/**
 * Plugin Name: TCR_YouTube_Gallery
 * Description: Shortcodes to show latest YouTube videos and recently updated playlists using public RSS (no API key).
 * Version:     1.2.3
 * Author:      The Code Rebel
 * Author URI:  https://thecoderebel.com
 * License:     GPL-2.0+
 * Text Domain: tcr-youtube-gallery
 */

if (!defined('ABSPATH')) { exit; }

class TCR_YouTube_Gallery {
    const OPT_KEY = 'tcr_youtube_gallery_settings';
    const TRANSIENT_PREFIX = 'tcr_yt_';
    const STYLE_HANDLE = 'tcr-yt-gallery';

    public function __construct() {
        // Admin
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_post_tcr_yt_clear_cache', [$this, 'handle_clear_cache']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_color_picker']);

        // Frontend
        add_action('wp_enqueue_scripts', [$this, 'register_front_styles']);
        add_shortcode('tcr_youtube_gallery', [$this, 'sc_gallery']);
        add_shortcode('tcr_youtube_playlists_recent', [$this, 'sc_playlists_recent']);
    }

    /* -------------------------
     * Admin
     * ------------------------- */
    public function register_admin_menu() {
        add_menu_page(
            'TCR YouTube Gallery',
            'TCR YouTube Gallery',
            'manage_options',
            'tcr-youtube-gallery',
            [$this, 'render_settings_page'],
            'dashicons-admin-generic',
            58
        );
        add_submenu_page(
            'tcr-youtube-gallery',
            'YouTube Gallery',
            'YouTube Gallery',
            'manage_options',
            'tcr-youtube-gallery',
            [$this, 'render_settings_page']
        );
    }

    public function enqueue_color_picker($hook) {
        if ($hook !== 'toplevel_page_tcr-youtube-gallery') return;
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }

    public function register_settings() {
        register_setting(self::OPT_KEY, self::OPT_KEY, [$this, 'sanitize_settings']);
        add_settings_section('tcr_yt_main', 'General', function(){
            echo '<p>Set your Channel ID, optional Playlist IDs, styling, and defaults.</p>';
        }, 'tcr-youtube-gallery');

        add_settings_field('channel_id', 'Channel ID', function(){
            $o = $this->get_options();
            echo '<input type="text" name="'.esc_attr(self::OPT_KEY).'[channel_id]" value="'.esc_attr($o['channel_id']).'" class="regular-text" placeholder="e.g. UC_xxx">';
        }, 'tcr-youtube-gallery', 'tcr_yt_main');

        add_settings_field('playlist_ids', 'Playlist IDs (optional)', function(){
            $o = $this->get_options();
            echo '<textarea name="'.esc_attr(self::OPT_KEY).'[playlist_ids]" rows="5" class="large-text" placeholder="PLxxxxxx, PLyyyyyy or one per line">'.esc_textarea($o['playlist_ids']).'</textarea>';
        }, 'tcr-youtube-gallery', 'tcr_yt_main');

        add_settings_field('border_color', 'Card Border Color', function(){
            $o = $this->get_options();
            echo '<input type="text" class="tcr-color-field" data-default-color="#e5e7eb" name="'.esc_attr(self::OPT_KEY).'[border_color]" value="'.esc_attr($o['border_color']).'" />';
            echo "<p class='description'>Pick a border color to match your theme.</p>";
            echo "<script>jQuery(function($){ $('.tcr-color-field').wpColorPicker(); });</script>";
        }, 'tcr-youtube-gallery', 'tcr_yt_main');

        add_settings_field('cache_ttl', 'Cache TTL (seconds)', function(){
            $o = $this->get_options();
            echo '<input type="number" min="60" step="60" name="'.esc_attr(self::OPT_KEY).'[cache_ttl]" value="'.esc_attr($o['cache_ttl']).'">';
        }, 'tcr-youtube-gallery', 'tcr_yt_main');

        add_settings_field('default_count', 'Default Video Count', function(){
            $o = $this->get_options();
            echo '<input type="number" min="1" max="50" name="'.esc_attr(self::OPT_KEY).'[default_count]" value="'.esc_attr($o['default_count']).'">';
        }, 'tcr-youtube-gallery', 'tcr_yt_main');
    }

    public function sanitize_settings($input) {
        $out = $this->get_options();
        $out['channel_id']    = sanitize_text_field($input['channel_id'] ?? '');
        $out['playlist_ids']  = $this->sanitize_playlist_ids($input['playlist_ids'] ?? '');
        $border               = sanitize_hex_color($input['border_color'] ?? '#e5e7eb');
        $out['border_color']  = $border ? $border : '#e5e7eb';
        $out['cache_ttl']     = max(60, intval($input['cache_ttl'] ?? 3600));
        $out['default_count'] = max(1, min(50, intval($input['default_count'] ?? 6)));
        return $out;
    }

    private function sanitize_playlist_ids($text) {
        $raw = preg_split('/[\\s,]+/', $text, -1, PREG_SPLIT_NO_EMPTY);
        $clean = [];
        foreach ($raw as $id) {
            $id = sanitize_text_field($id);
            if ($id && !in_array($id, $clean, true)) $clean[] = $id;
        }
        return implode("\n", $clean);
    }

    private function get_options() {
        $defaults = [
            'channel_id'    => '',
            'playlist_ids'  => '',
            'border_color'  => '#e5e7eb',
            'cache_ttl'     => 3600,
            'default_count' => 6,
        ];
        return wp_parse_args(get_option(self::OPT_KEY, []), $defaults);
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) return;
        $o = $this->get_options();
        $tracked = array_filter(array_map('trim', explode("\n", $o['playlist_ids'])));
        $recent = $tracked ? $this->get_recent_playlists($tracked, $o['cache_ttl']) : [];

        echo '<div class="wrap"><h1>YouTube Gallery</h1>';
        echo '<form method="post" action="options.php">';
        settings_fields(self::OPT_KEY);
        do_settings_sections('tcr-youtube-gallery');
        submit_button();
        $url = wp_nonce_url(admin_url('admin-post.php?action=tcr_yt_clear_cache'), 'tcr_yt_clear_cache');
        echo ' <a href="'.esc_url($url).'" class="button">Clear Cache</a>';
        echo '</form>';

        // Copyable shortcodes
        $short1 = '[tcr_youtube_gallery count="6" show_title="true" new_tab="true" thumb="hq"]';
        $short2 = '[tcr_youtube_gallery playlist_id="PLxxxxxxxxxxxx" count="8"]';
        $short3 = '[tcr_youtube_playlists_recent limit="6" show_last_video="true"]';
        echo '<hr><h2>Shortcodes</h2>';
        foreach ([$short1,$short2,$short3] as $s) {
            echo '<p><input type="text" class="large-text code" readonly value="'.esc_attr($s).'" onfocus="this.select();"></p>';
        }

        // Recently Updated Playlists table
        echo '<hr><h2>Recently Updated Playlists</h2>';
        if (!$tracked) {
            echo '<p>Add Playlist IDs above to populate this list.</p></div>'; return;
        }
        if (empty($recent)) { echo '<p>No data yet. Try again later.</p></div>'; return; }
        echo '<table class="widefat striped"><thead><tr><th>Playlist</th><th>Last Video</th><th>Updated</th><th>Link</th></tr></thead><tbody>';
        foreach ($recent as $row) {
            $plink = 'https://www.youtube.com/playlist?list=' . rawurlencode($row['playlist_id']);
            $vlink = $row['last_video_id'] ? 'https://www.youtube.com/watch?v=' . rawurlencode($row['last_video_id']) : '#';
            echo '<tr>';
            echo '<td><strong>'.esc_html($row['playlist_title']).'</strong><br><code>'.esc_html($row['playlist_id']).'</code></td>';
            echo '<td>'.($row['last_video_id']?'<a href="'.esc_url($vlink).'" target="_blank" rel="noopener">'.esc_html($row['last_video_title']).'</a>':'—').'</td>';
            echo '<td>'.esc_html($row['last_updated_human']).'</td>';
            echo '<td><a class="button button-small" href="'.esc_url($plink).'" target="_blank" rel="noopener">Open</a></td>';
            echo '</tr>';
        }
        echo '</tbody></table></div>';
    }

    public function handle_clear_cache() {
        if (!current_user_can('manage_options')) wp_die('Nope.');
        check_admin_referer('tcr_yt_clear_cache');
        global $wpdb;
        $like = $wpdb->esc_like(self::TRANSIENT_PREFIX);
        $wpdb->query( $wpdb->prepare("DELETE FROM $wpdb->options WHERE option_name LIKE %s OR option_name LIKE %s", '_transient_'.$like.'%', '_transient_timeout_'.$like.'%') );
        wp_safe_redirect( admin_url('admin.php?page=tcr-youtube-gallery&cleared=1') );
        exit;
    }

    /* -------------------------
     * Frontend
     * ------------------------- */
    public function register_front_styles() {
        $o = $this->get_options();
        $border = esc_attr($o['border_color']);
        $css = "
        .tcr-yt-grid{display:grid;grid-template-columns:repeat(auto-fill,minmax(220px,1fr));gap:16px;align-items:start}
        .tcr-yt-card{display:block;text-decoration:none;border-radius:12px;overflow:hidden;background:#fff;border:2px solid {$border};box-shadow:0 1px 2px rgba(0,0,0,.04);transition:transform .12s ease, box-shadow .12s ease}
        .tcr-yt-card:focus,.tcr-yt-card:hover{transform:translateY(-1px);box-shadow:0 6px 16px rgba(0,0,0,.08)}
        .tcr-yt-thumb{aspect-ratio:16/9;width:100%;height:auto;display:block;object-fit:cover;background:#f3f4f6}
        .tcr-yt-title{font-size:14px;line-height:1.35;font-weight:600;color:#111827;margin:8px 10px 10px}
        .tcr-yt-title:hover{text-decoration:underline}
        .tcr-yt-sr-only{position:absolute;width:1px;height:1px;padding:0;margin:-1px;overflow:hidden;clip:rect(0,0,0,0);white-space:nowrap;border:0}
        ";
        wp_register_style(self::STYLE_HANDLE, false);
        wp_add_inline_style(self::STYLE_HANDLE, $css);
    }

    public function sc_gallery($atts) {
        $o = $this->get_options();
        $atts = shortcode_atts([
            'channel_id'  => $o['channel_id'],
            'playlist_id' => '',
            'count'       => $o['default_count'],
            'show_title'  => 'true',
            'new_tab'     => 'true',
            'thumb'       => 'hq', // mq, hq, sd, max
            'class'       => '',
        ], $atts, 'tcr_youtube_gallery');

        $count = max(1, min(50, intval($atts['count'])));
        $show_title = filter_var($atts['show_title'], FILTER_VALIDATE_BOOLEAN);
        $new_tab = filter_var($atts['new_tab'], FILTER_VALIDATE_BOOLEAN);
        $thumb_quality = in_array($atts['thumb'], ['mq','hq','sd','max'], true) ? $atts['thumb'] : 'hq';

        $feed_url = '';
        if (!empty($atts['playlist_id'])) {
            $feed_url = 'https://www.youtube.com/feeds/videos.xml?playlist_id=' . rawurlencode($atts['playlist_id']);
        } elseif (!empty($atts['channel_id'])) {
            $feed_url = 'https://www.youtube.com/feeds/videos.xml?channel_id=' . rawurlencode($atts['channel_id']);
        } else {
            return '<p>Please set a Channel ID in WP-Admin (The Code Rebel → YouTube Gallery) or pass <code>playlist_id</code> to the shortcode.</p>';
        }

        wp_enqueue_style(self::STYLE_HANDLE);

        $cache_key = self::TRANSIENT_PREFIX . 'latest_' . md5($feed_url . '|' . $count);
        $items = get_transient($cache_key);
        if ($items === false) {
            $items = $this->fetch_feed_items($feed_url, $count);
            if (is_wp_error($items)) {
                return '<p>Failed to load YouTube feed. ' . esc_html($items->get_error_message()) . '</p>';
            }
            set_transient($cache_key, is_array($items) ? $items : [], $this->get_options()['cache_ttl']);
        }

        if (!$items) return '<p>No videos found.</p>';

        $target = $new_tab ? ' target="_blank" rel="noopener nofollow ugc"' : '';
        $wrapper_classes = 'tcr-yt-grid' . (!empty($atts['class']) ? ' ' . esc_attr($atts['class']) : '');

        ob_start();
        echo '<div class="' . $wrapper_classes . '">';
        foreach ($items as $it) {
            $video_id = $it['video_id'];
            $title    = $it['title'];
            $video_url= 'https://www.youtube.com/watch?v=' . rawurlencode($video_id);
            $thumb_url= $this->thumb_url($video_id, $thumb_quality);

            echo '<a class="tcr-yt-card" href="' . esc_url($video_url) . '"' . $target . ' aria-label="' . esc_attr($title) . '">';
            echo '<img class="tcr-yt-thumb" loading="lazy" src="' . esc_url($thumb_url) . '" alt="' . esc_attr($title) . ' thumbnail" />';
            if ($show_title) {
                echo '<div class="tcr-yt-title">' . esc_html($title) . '</div>';
            } else {
                echo '<span class="tcr-yt-sr-only">' . esc_html($title) . '</span>';
            }
            echo '</a>';
        }
        echo '</div>';
        return ob_get_clean();
    }

    public function sc_playlists_recent($atts) {
        $o = $this->get_options();
        $tracked = array_filter(array_map('trim', explode("\n", $o['playlist_ids'])));

        $atts = shortcode_atts([
            'limit' => 10,
            'show_last_video' => 'true',
            'new_tab' => 'true',
        ], $atts, 'tcr_youtube_playlists_recent');

        $limit = max(1, intval($atts['limit']));
        $show_last_video = filter_var($atts['show_last_video'], FILTER_VALIDATE_BOOLEAN);
        $new_tab = filter_var($atts['new_tab'], FILTER_VALIDATE_BOOLEAN);
        $target = $new_tab ? ' target="_blank" rel="noopener nofollow ugc"' : '';

        if (!$tracked) return '<p>No tracked playlists. Add some in WP-Admin → The Code Rebel → YouTube Gallery.</p>';

        $rows = $this->get_recent_playlists($tracked, $o['cache_ttl']);
        if (!$rows) return '<p>No playlist data available.</p>';

        $rows = array_slice($rows, 0, $limit);

        ob_start();
        echo '<div class="tcr-yt-playlists">';
        echo '<ul style="list-style:none;padding:0;margin:0;display:grid;grid-template-columns:repeat(auto-fill,minmax(260px,1fr));gap:16px;">';
        foreach ($rows as $r) {
            $plink = 'https://www.youtube.com/playlist?list=' . rawurlencode($r['playlist_id']);
            $vlink = $r['last_video_id'] ? 'https://www.youtube.com/watch?v=' . rawurlencode($r['last_video_id']) : '#';

            echo '<li style="border:2px solid '.esc_attr($o['border_color']).';border-radius:12px;padding:12px;background:#fff;box-shadow:0 1px 2px rgba(0,0,0,.04)">';
            echo '<div style="font-weight:700;margin-bottom:6px;"><a href="'.esc_url($plink).'"'.$target.'>'.esc_html($r['playlist_title']).'</a></div>';
            echo '<div style="font-size:12px;color:#6b7280;margin-bottom:8px;">Updated '.$r['last_updated_human'].'</div>';
            if ($show_last_video && $r['last_video_id']) {
                echo '<div style="font-size:14px;"><a href="'.esc_url($vlink).'"'.$target.'>'.esc_html($r['last_video_title']).'</a></div>';
            }
            echo '</li>';
        }
        echo '</ul></div>';

        return ob_get_clean();
    }

    /* -------------------------
     * Helpers
     * ------------------------- */
    private function fetch_feed_items($feed_url, $limit) {
        $resp = wp_remote_get($feed_url, [
            'timeout' => 10,
            'headers' => ['Accept' => 'application/atom+xml'],
            'user-agent' => 'TCR-YT/1.2.3 (+https://wordpress.org/)',
        ]);
        if (is_wp_error($resp)) return $resp;

        $code = wp_remote_retrieve_response_code($resp);
        if ($code !== 200) {
            return new WP_Error('tcr_bad_status', 'HTTP ' . $code . ' fetching feed.');
        }

        $body = wp_remote_retrieve_body($resp);
        if (!$body) return new WP_Error('tcr_empty', 'Empty response body.');

        $xml = @simplexml_load_string($body);
        if ($xml === false) return new WP_Error('tcr_parse', 'Failed to parse XML.');

        $xml->registerXPathNamespace('atom', 'http://www.w3.org/2005/Atom');
        $xml->registerXPathNamespace('yt', 'http://www.youtube.com/xml/schemas/2015');

        $entries = $xml->xpath('//atom:entry');
        $out = [];
        foreach ($entries as $entry) {
            $ns = $entry->getNamespaces(true);
            $yt = $ns['yt'] ?? null;

            $video_id = '';
            if ($yt && isset($entry->children($yt)->videoId)) {
                $video_id = (string)$entry->children($yt)->videoId;
            } else {
                foreach ($entry->link as $lnk) {
                    $href = (string) $lnk['href'];
                    if (strpos($href, 'watch?v=') !== false) {
                        $parts = wp_parse_url($href);
                        if (!empty($parts['query'])) {
                            parse_str($parts['query'], $q);
                            if (!empty($q['v'])) $video_id = sanitize_text_field($q['v']);
                        }
                        break;
                    }
                }
            }
            if (!$video_id) continue;

            $title = (string)$entry->title;
            $out[] = [
                'video_id' => sanitize_text_field($video_id),
                'title'    => sanitize_text_field($title),
                'published'=> (string)($entry->published ?? ''),
            ];
            if (count($out) >= $limit) break;
        }
        return $out;
    }

    private function thumb_url($video_id, $quality) {
        $map = ['mq'=>'mqdefault.jpg','hq'=>'hqdefault.jpg','sd'=>'sddefault.jpg','max'=>'maxresdefault.jpg'];
        $file = isset($map[$quality]) ? $map[$quality] : $map['hq'];
        return sprintf('https://i.ytimg.com/vi/%s/%s', rawurlencode($video_id), $file);
    }

    private function playlist_feed_url($playlist_id) {
        return 'https://www.youtube.com/feeds/videos.xml?playlist_id=' . rawurlencode($playlist_id);
    }

    private function get_recent_playlists(array $playlist_ids, $ttl) {
        $rows = [];
        foreach ($playlist_ids as $pid) {
            $pid_clean = sanitize_text_field($pid);
            $key = self::TRANSIENT_PREFIX . 'pl_' . md5($pid_clean);
            $data = get_transient($key);
            if ($data === false) {
                $data = $this->fetch_playlist_meta($pid_clean);
                if (!is_wp_error($data)) {
                    set_transient($key, $data, $ttl);
                }
            }
            if (is_wp_error($data) || empty($data)) continue;
            $rows[] = $data;
        }

        usort($rows, function($a, $b){
            return strtotime($b['last_updated']) <=> strtotime($a['last_updated']);
        });

        foreach ($rows as &$r) {
            $r['last_updated_human'] = $this->human_time_diff_str(strtotime($r['last_updated']));
        }
        return $rows;
    }

    private function fetch_playlist_meta($playlist_id) {
        $url = $this->playlist_feed_url($playlist_id);
        $resp = wp_remote_get($url, [
            'timeout' => 10,
            'headers' => ['Accept' => 'application/atom+xml'],
            'user-agent' => 'TCR-YT/1.2.3 (+https://wordpress.org/)',
        ]);
        if (is_wp_error($resp)) return $resp;
        $code = wp_remote_retrieve_response_code($resp);
        if ($code !== 200) return new WP_Error('tcr_bad_status', 'HTTP ' . $code . ' fetching playlist feed.');
        $body = wp_remote_retrieve_body($resp);
        if (!$body) return new WP_Error('tcr_empty', 'Empty response body.');

        $xml = @simplexml_load_string($body);
        if ($xml === false) return new WP_Error('tcr_parse', 'Failed to parse XML.');

        $xml->registerXPathNamespace('atom', 'http://www.w3.org/2005/Atom');
        $xml->registerXPathNamespace('yt', 'http://www.youtube.com/xml/schemas/2015');

        $pl_title = (string)($xml->title ?? ('Playlist ' . $playlist_id));

        $entry = $xml->xpath('//atom:entry')[0] ?? null;
        if (!$entry) {
            return [
                'playlist_id' => $playlist_id,
                'playlist_title' => sanitize_text_field($pl_title),
                'last_updated' => '1970-01-01T00:00:00Z',
                'last_video_id' => '',
                'last_video_title' => 'No videos',
            ];
        }

        $ns = $entry->getNamespaces(true);
        $yt = $ns['yt'] ?? null;

        $last_video_id = '';
        if ($yt && isset($entry->children($yt)->videoId)) {
            $last_video_id = (string)$entry->children($yt)->videoId;
        }

        $last_title = (string)$entry->title;
        $last_published = (string)($entry->published ?? '');

        return [
            'playlist_id' => $playlist_id,
            'playlist_title' => sanitize_text_field($pl_title),
            'last_updated' => $last_published ?: gmdate('c'),
            'last_video_id' => sanitize_text_field($last_video_id),
            'last_video_title' => sanitize_text_field($last_title),
        ];
    }

    private function human_time_diff_str($ts) {
        if (!$ts) return 'unknown';
        if ($ts > time()) $ts = time();
        $diff = time() - $ts;
        if ($diff < MINUTE_IN_SECONDS) return 'just now';
        return human_time_diff($ts, time()) . ' ago';
    }
}

new TCR_YouTube_Gallery();
